﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using BrightIdeasSoftware;
using ExtensionMethods;
using AutoUpdaterDotNET;
using System.Net;

namespace GUSLogReader
{
    

    public partial class frmMain : Form
    {
        public String VERSION = "";
        public static String SOURCEURL = @"https://gusapps.ecprojects.de/logreader/GUSLogReader-Source.zip";
        private const String UPDATESERVER = @"https://gusapps.ecprojects.de/logreader/update.xml";


        private Logfiles currentLogSet;
        private String currentSetFile;
        private List<MSG> lastLoadedObjects;
        private List<MSG> lastFilteredObjects;

        private RecentFilesHelper recentOpenedFiles;

        private SourceFilter FilterSet;

        Object lockMessageList = new Object();
        Object lockProgressCtr = new Object();

        public delegate void ProgressChangeDelegate(int currentValue, int maxValue, String currentFile);
        public event ProgressChangeDelegate OnProgressChanged;

        private System.Diagnostics.Stopwatch watch;

        private Color Color1;
        private Color Color2;
        private Color Color3;

        private Dictionary<string, Color> HashColors;

        public void ChangeProgressState(int currentValue, int maxValue, String currentFile)
        {
            this.toolStripProgressBar1.Maximum = maxValue;
            if (currentValue <= maxValue)
                this.toolStripProgressBar1.Value = currentValue;

            this.toolStripCurrentFile.Text = String.Format("Verarbeite: {0}", currentFile);

            this.toolStripProgressBar1.Invalidate();
            this.toolStripCurrentFile.Invalidate();
        }

        public frmMain()
        {
            InitializeComponent();
            this.VERSION = Application.ProductVersion;
            this.Text = "GUS Log Reader - V" + VERSION;

            currentLogSet = null;
            OnProgressChanged += ChangeProgressState;
            setUI(false);
            currentSetFile = "";
            FilterSet = new SourceFilter();



            Color1 = Color.FromArgb(196, 89, 17);
            Color2 = Color.FromArgb(112, 173, 71);
            Color3 = Color.FromArgb(142, 170, 219);

            HashColors = new Dictionary<string, Color>();


            watch = new System.Diagnostics.Stopwatch();
            recentOpenedFiles = new RecentFilesHelper(Application.StartupPath);
            LoadRecentFileList();

            RefreshMarkColor();

            AutoUpdater.CheckForUpdateEvent += AutoUpdaterOnCheckForUpdateEvent;
            CheckUpdates();

        }

        private void CheckUpdates()
        {
            this.resultStatusLabel.Text = "Prüfe auf Updates...";

            //check if updateserver is aviable
            WebRequest request = WebRequest.Create(UPDATESERVER);


            HttpWebResponse response = (HttpWebResponse)request.GetResponse();
            if (response == null || response.StatusCode != HttpStatusCode.OK)
            {
                resultStatusLabel.Text = "Updateserver nicht erreichbar.";
            }
            else
            {                
                AutoUpdater.Start(UPDATESERVER);               
            }
        }


        private void AutoUpdaterOnCheckForUpdateEvent(UpdateInfoEventArgs args)
        {
            if (args != null)
            {
                if (args.IsUpdateAvailable)
                {
                    DialogResult dialogResult;
                    if (args.Mandatory)
                    {
                        dialogResult = MessageBox.Show(
                                $@"Die neue Version {args.CurrentVersion} ist verfügbar. Sie verwenden derzeit die Version {args.InstalledVersion}. Klicken Sie OK um mit dem Update zu beginnen.", @"Update verfügbar!",
                                MessageBoxButtons.OK,
                                MessageBoxIcon.Information);
                    }
                    else
                    {
                        dialogResult =
                            MessageBox.Show(
                                $@"Die neue Version {args.CurrentVersion} ist verfügbar. Sie verwenden derzeit die Version {args.InstalledVersion}. Wollen Sie das Update jetzt installieren?", @"Update verfügbar!",
                                MessageBoxButtons.YesNo,
                                MessageBoxIcon.Information);
                    }

                    if (dialogResult.Equals(DialogResult.Yes))
                    {
                        try
                        {
                            if (AutoUpdater.DownloadUpdate())
                            {
                                Application.Exit();
                            }
                        }
                        catch (Exception exception)
                        {
                            MessageBox.Show(exception.Message, exception.GetType().ToString(), MessageBoxButtons.OK,
                                MessageBoxIcon.Error);
                        }
                    }
                }
                else
                {
                    resultStatusLabel.Text = @"Es ist derzeit kein Update verfügbar.";
                }
            }
            else
            {
                MessageBox.Show(
                        @"There is a problem reaching update server please check your internet connection and try again later.",
                        @"Update check failed", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void loadFileFromRecent(object sender, EventArgs e)
        {
            var a = (RibbonButton)sender;
            LoadFile(a.Tag.ToString());
        }
        private void LoadRecentFileList()
        {
            ribbon1.OrbDropDown.RecentItems.Clear();
            foreach (var item in recentOpenedFiles.files.GetFiles())
            {
                var a = new RibbonButton();
                a.Click += loadFileFromRecent;
                a.Text = item;
                a.Tag = item;
                ribbon1.OrbDropDown.RecentItems.Add(a);
            }
            
        }
        private void NewFileSet()
        {
            //New Set
            currentLogSet = new Logfiles();
            if (SaveFileAs())
              setUI(true);
        }
        private void LogTypeDialog()
        {
            //LogTypes
            if (!(currentLogSet == null))
            {
                var f = new frmLogTypes(currentLogSet.Prefixes);
                f.ShowDialog();
                currentLogSet.Prefixes = f.getLogTypes();
            }
        }

        private bool SaveFileAs()
        {
            //Save as
            var fsd = new SaveFileDialog();
            fsd.Filter = "Log Sammlungen (*.lsm)|*.lsm";

            if (fsd.ShowDialog() == DialogResult.OK)
            {
                
                currentSetFile = fsd.FileName;
                toolStripStatusLabel1.Text = currentSetFile;
                recentOpenedFiles.files.AddFile(currentSetFile);
                LoadRecentFileList();
                save();
                return true;
            }
            return false;
        }
        private void Exit()
        {

            this.Close();
        }
        private void LoadFile(String path)
        {
            currentLogSet = XMLHelper.DeSerialize<Logfiles>(System.IO.File.ReadAllText(path));
            currentSetFile = path;
            this.toolStripStatusLabel1.Text = path;

            recentOpenedFiles.files.AddFile(currentSetFile);
            LoadRecentFileList();

            if ((currentLogSet!=null) && (!currentLogSet.ListViewConfig.Equals("")))
                olv.RestoreState(XMLHelper.DeSerializeAtomic<byte[]>(currentLogSet.ListViewConfig));

            

            this.WindowState = currentLogSet.LastWindowState;

            setUI(true);
            ReLoadElements();

            this.FilterSet = currentLogSet.filters;
            LoadFilterCombos();


        }
        private void LoadFile()
        {
            //Load
            var ofd = new OpenFileDialog();
            ofd.Filter = "Log Sammlungen (*.lsm)|*.lsm";

            if (ofd.ShowDialog() == DialogResult.OK)
            {
                LoadFile(ofd.FileName);
            }
        }

        private void ReFreshElements(bool clearOnly=false)
        {
            this.txtDetails.Text = "";
            this.rtSearch.TextBoxText = "";
            this.rbuttonSearch.Checked = false;
            this.ribbonButtonMarkieren.Checked = false;

            olv.AdditionalFilter = null;
            this.olv.DefaultRenderer = new BaseRenderer();
            this.olv.ModelFilter = null;

            FilterSet.ExcludedHeadLines.Clear();
            ribbonComboHeadFilter.DropDownItems.Clear();
            ribbonComboHeadFilter.TextBoxText = "";
            

            FilterSet.ExcludedSourceFiles.Clear();
            ribbonComboFilterSource.DropDownItems.Clear();
            ribbonComboFilterSource.TextBoxText = "";

            FilterSet.ExcludedKategories.Clear();
            ribbonComboFilterKateg.DropDownItems.Clear();
            ribbonComboFilterKateg.TextBoxText = "";

            FilterSet.ExcludedClasses.Clear();
            ribbonComboFilterKlasse.DropDownItems.Clear();
            ribbonComboFilterKlasse.TextBoxText = "";


            CalculateHashColors();

            if (!clearOnly)
                olv.SetObjects(this.lastLoadedObjects);
        }
        private void LoadFilterCombos()
        {

            this.ribbonComboFilterKateg.DropDownItems.Clear();
            foreach (String item in FilterSet.ExcludedKategories)
            {
                var i = new RibbonButton();
                i.Text = item;
                i.ToolTip = "Klicken um den Filter zu entfernen";
                i.Click += removeKategorieFilter;

                ribbonComboFilterKateg.DropDownItems.Add(i);
            }

            this.ribbonComboFilterKlasse.DropDownItems.Clear();
            foreach (String item in FilterSet.ExcludedClasses)
            {
                var i = new RibbonButton();
                i.Text = item;
                i.ToolTip = "Klicken um den Filter zu entfernen";
                i.Click += removeClassFilter;

                ribbonComboFilterKlasse.DropDownItems.Add(i);
            }

            this.ribbonComboFilterSource.DropDownItems.Clear();
            foreach (String item in FilterSet.ExcludedSourceFiles)
            {
                var i = new RibbonButton();
                i.Text = item;
                i.ToolTip = "Klicken um den Filter zu entfernen";
                i.Click += removeSourceFilter;

                ribbonComboFilterSource.DropDownItems.Add(i);
            }

            this.ribbonComboHeadFilter.DropDownItems.Clear();
            foreach (String item in FilterSet.ExcludedHeadLines)
            {
                var i = new RibbonButton();
                i.Text = item;
                i.ToolTip = "Klicken um den Filter zu entfernen";
                i.Click += removeHeadLineFilter;

                ribbonComboHeadFilter.DropDownItems.Add(i);
            }
        }
        private void LoadLogElements(object sender, DoWorkEventArgs e)
        {
            int all=0, cur = 0;
            var now = DateTime.Now;
            

            var fileList = new List<string>();
            foreach (var path in currentLogSet.Path)
            {
                foreach (var prefix in currentLogSet.Prefixes)
                {
                    var files = Logfiles.RecursiveDirSearch(path, prefix);

                    foreach (var file in files)
                    {
                        if (!(fileList.Contains(file)))
                         {
                             if (currentLogSet.CheckFileMaxAge)
                             {
                                 var filetime = System.IO.File.GetLastWriteTime(file);
                                 var fileage = now - filetime;
                                 if (fileage.TotalDays <= currentLogSet.FileMaxAge)
                                     fileList.Add(file);
                             }
                             else
                             {
                                 fileList.Add(file);
                             }
                         }
                     }
                }
            }


            all = fileList.Count;

            var msgList = new List<MSG>();
            //foreach (var item in fileList)
            Parallel.ForEach(fileList, item =>
             {
                 lock (lockProgressCtr)
                 {
                     cur++;
                 }


                 //YES, this is not thread-save but it isn't really important which file of threads is processd
                 this.Invoke(OnProgressChanged, cur, all, item);

                 var res = MsgReader.processFile(item, currentLogSet.SameTimeValue);

                 lock (lockMessageList)
                 {
                     msgList.AddRange(res);
                 }

             });
            e.Result = msgList;
        }
        private void bg_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            // Retrieve the result pass from bg_DoWork() if any.
            // Note, you may need to cast it to the desired data type.
            var objs = (List<MSG>)e.Result;
            this.lastLoadedObjects = objs;
            olv.SetObjects(objs);

            this.toolStripProgressBar1.Value = 0;
            this.toolStripCurrentFile.Text = "";
            this.toolStripProgressBar1.Visible = false;

            this.watch.Stop();

            resultStatusLabel.Text = String.Format("{1} Elemente geladen in: {0} s", watch.Elapsed.TotalSeconds,objs.Count);
            CalculateHashColors();
            ApplyFilterSet();

            olv.Sort(olvColumn1);
            // Close the loading form.
            //loadingForm.Close();

            // Update any other UI controls that may need to be updated.
        }
        private void ribbonButton1_Click(object sender, EventArgs e)
        {
            if (currentLogSet != null)
                ReLoadElements();
        }

        private void ReLoadElements(bool reFresh = true)
        {
            if (reFresh)
                ReFreshElements(true);

            this.toolStripProgressBar1.Visible = true;
            resultStatusLabel.Text = "";

            this.watch.Reset();
            this.watch.Start();

            BackgroundWorker bg = new BackgroundWorker();
            bg.DoWork += new DoWorkEventHandler(LoadLogElements);
            bg.RunWorkerCompleted += new RunWorkerCompletedEventHandler(bg_RunWorkerCompleted);
            bg.RunWorkerAsync();
            
            
        }
        private void ExportSelectedToText()
        {
            String text = "";

            if (olv.SelectedObjects != null)
            {
                foreach (MSG item in olv.SelectedObjects)
                {
                    text += String.Format(currentLogSet.TextFormatLine, item.TS, item.Kat, item.Thread, item.Klasse,  item.Message);                    
                    text += Environment.NewLine;
                }

                var f = new frmText();
                f.Text = String.Format("LOG-Text {0:yyyy-MM-dd HH:mm:ss,fff}", ((MSG)olv.SelectedObjects[0]).TS);
                if (olv.SelectedObjects.Count > 1)
                    f.Text += String.Format(" ({0} Einträge)",olv.SelectedObjects.Count);
                f.setText(text);
                f.Show();
            }

        }
        private void ExportSelectedToCSV()
        {
            if (olv.SelectedObjects != null)
            {
                string csv = string.Empty;

                OLVExporter olvExporter;

                if ((olv.SelectedObjects != null) && (olv.SelectedObjects.Count > 0))
                   olvExporter = new OLVExporter(olv,olv.SelectedObjects);
                else
                    olvExporter = new OLVExporter(olv, olv.FilteredObjects);


                csv = olvExporter.ExportTo(OLVExporter.ExportFormat.CSV);
                csv = csv.Replace(",", ";");

                SaveFileDialog saveFile = new SaveFileDialog();
                saveFile.FileName = "ExportFile.csv";
                saveFile.Filter = "csv files (*.csv)|*.csv";

                if (saveFile.ShowDialog() == DialogResult.OK)
                {
                    using (System.IO.StreamWriter sw = new System.IO.StreamWriter(saveFile.FileName, false, Encoding.UTF8))
                    {
                        sw.Write(csv);
                    }
                }
            }
        }
        private void LoadSelectedObjectDetails()
        {
            txtDetails.Text = "";
            if (olv.SelectedObjects != null)
            {
                foreach (var item in olv.SelectedObjects)
                {
                    var elem = (MSG)item;
                    txtDetails.Text += elem.Message + Environment.NewLine;
                }
                
            }
        }
        private void MarkSelectedColor(int colorID)
        {
            if (olv.SelectedObjects != null)
            {
                foreach (var item in olv.SelectedObjects)
                {
                    var elem = (MSG)item;
                    if (elem.MarkerID == 0)
                        elem.MarkerID = colorID;
                    else
                        elem.MarkerID = 0;
                }
            }
            olv.RefreshObjects(olv.SelectedObjects);
        }
        private void LoadSelectionMarkerState()
        {
            if (olv.SelectedObjects != null)
            {
                bool isOneC1 = false;
                bool isOneC2 = false;
                bool isOneC3 = false;

                foreach (var item in olv.SelectedObjects)
                {
                    var elem = (MSG)item;
                    if (elem.MarkerID == 1)
                        isOneC1 = true;

                    if (elem.MarkerID == 2)
                        isOneC2 = true;

                    if (elem.MarkerID == 3)
                        isOneC3 = true;
                }
                ribbonButtonMark1.Checked = isOneC1;
                ribbonButtonMark2.Checked = isOneC2;
                ribbonButtonMark3.Checked = isOneC3;

            }
        }
        private void olv_Click(object sender, EventArgs e)
        {
            LoadSelectedObjectDetails();
        }

        private void save()
        {
            if ((currentSetFile != "") && (currentLogSet != null))
            {
                currentLogSet.LastWindowState = this.WindowState;
                currentLogSet.ListViewConfig = XMLHelper.Serialize<byte[]>(olv.SaveState());
                currentLogSet.filters = FilterSet;
                System.IO.File.WriteAllText(currentSetFile, XMLHelper.Serialize(currentLogSet));
            }
        }
        private void setUI(bool enabled)
        {
            ribbonPanel2.Enabled = enabled;
            ribbonOrbMenuItem3.Enabled = enabled;
            ribbonOrbMenuItem4.Enabled = enabled;

            ribbonOrbOptionButton1.Enabled = enabled;
            ribbonOrbOptionButton2.Enabled = enabled;
            ribbonOrbOptionOpts.Enabled = enabled;

            ribbonPanel3.Enabled = enabled;
            ribbonPanel5.Enabled = enabled;
            ribbonPanel6.Enabled = enabled;
            ribbonPanel8.Enabled = enabled;

            //Design
            ribbonCheckBox1.Checked = olv.UseAlternatingBackColors;
            ribbonCheckBox2.Checked = olv.GridLines;
            ribbonCheckBox3.Checked = olv.FullRowSelect;

        }

        private void LogFilePathDialog()
        {
            if (!(currentLogSet == null))
            {
                var f = new frmLogPath(currentLogSet);
                f.ShowDialog();
                currentLogSet.Path = f.getLogPath();
            }
        }
        private void ApplyFilterSet()
        {
            this.lastFilteredObjects = new List<MSG>(this.lastLoadedObjects);

            foreach (var item in FilterSet.ExcludedHeadLines)
            {
                lastFilteredObjects.RemoveAll(x => x.Msg1stL == item);
            }

            foreach (var item in FilterSet.ExcludedSourceFiles)
            {
                lastFilteredObjects.RemoveAll(x => x.Source == item);
            }

            foreach (var item in FilterSet.ExcludedKategories)
            {
                lastFilteredObjects.RemoveAll(x => x.Kat == item);
            }

            foreach (var item in FilterSet.ExcludedClasses)
            {
                lastFilteredObjects.RemoveAll(x => x.Klasse == item);
            }

            if (ribbonCheckOnlyToday.Checked)
            {
                lastFilteredObjects.RemoveAll(x => x.TS.IsSameDay(DateTime.Today) == false);
            }

            if (ribbonCheckSelectC1.Checked || ribbonCheckSelectC2.Checked || ribbonCheckSelectC3.Checked)
            {
                lastFilteredObjects.RemoveAll(x => x.MarkerID == 0);

                if (!ribbonCheckSelectC1.Checked)
                    lastFilteredObjects.RemoveAll(x => x.MarkerID == 1);

                if (!ribbonCheckSelectC2.Checked)
                    lastFilteredObjects.RemoveAll(x => x.MarkerID == 2);

                if (!ribbonCheckSelectC3.Checked)
                    lastFilteredObjects.RemoveAll(x => x.MarkerID == 3);
            }


            olv.SetObjects(lastFilteredObjects);

        }
        private void rbtnSearch_Click(object sender, EventArgs e)
        {
            rtSearch_TextBoxTextChanged(sender, e);
        }

        private void rtSearch_TextBoxTextChanged(object sender, EventArgs e)
        {
            TextMatchFilter filter = TextMatchFilter.Contains(this.olv, rtSearch.TextBoxText);
            if (rbuttonSearch.Checked)
                this.olv.AdditionalFilter = filter;
            else
                this.olv.AdditionalFilter = null;

            if (ribbonButtonMarkieren.Checked)
            {
                this.olv.DefaultRenderer = new HighlightTextRenderer(filter);
                this.olv.Invalidate();
            }
            else
                this.olv.DefaultRenderer = new BaseRenderer();
        }
        private void removeHeadLineFilter(object sender, EventArgs e)
        {
            var i = (RibbonButton)sender;
            this.FilterSet.ExcludedHeadLines.Remove(i.Text);
            this.ribbonComboHeadFilter.DropDownItems.Remove(i);
            this.ribbonComboHeadFilter.TextBoxText = "";
            ApplyFilterSet();
        }
        private void removeKategorieFilter(object sender, EventArgs e)
        {
            var i = (RibbonButton)sender;
            this.FilterSet.ExcludedKategories.Remove(i.Text);
            this.ribbonComboFilterKateg.DropDownItems.Remove(i);
            this.ribbonComboFilterKateg.TextBoxText = "";
            ApplyFilterSet();
        }
        private void removeClassFilter(object sender, EventArgs e)
        {
            var i = (RibbonButton)sender;
            this.FilterSet.ExcludedClasses.Remove(i.Text);
            this.ribbonComboFilterKlasse.DropDownItems.Remove(i);
            this.ribbonComboFilterKlasse.TextBoxText = "";
            ApplyFilterSet();
        }
        private void removeSourceFilter(object sender, EventArgs e)
        {
            var i = (RibbonButton)sender;
            this.FilterSet.ExcludedSourceFiles.Remove(i.Text);
            this.ribbonComboFilterSource.DropDownItems.Remove(i);
            this.ribbonComboFilterSource.TextBoxText = "";
            ApplyFilterSet();
        }
        private void colorizeCellCheck()
        {
            if (ribbonCheckUseColorCellsKat.Checked || ribbonCheckColorClass.Checked || ribbonCheckColorSource.Checked || ribbonCheckColorThread.Checked)
            {
                olv.UseCellFormatEvents = true;
                this.olv.FormatCell += ColorizeCells;                
            }
            else
            {
                olv.UseCellFormatEvents = false;
                this.olv.FormatCell -= ColorizeCells;
            }
            this.olv.Invalidate();
        }

    private void ribbonOrbMenuItem1_Click_1(object sender, EventArgs e)
        {
            NewFileSet();
        }

        private void ribbonOrbMenuItem2_Click(object sender, EventArgs e)
        {
            LoadFile();
        }

        private void ribbonOrbMenuItem3_Click(object sender, EventArgs e)
        {
            save();
        }

        private void ribbonOrbMenuItem4_Click(object sender, EventArgs e)
        {
            SaveFileAs();
        }

        private void ribbonOrbMenuItem5_Click(object sender, EventArgs e)
        {
            Exit();
        }

        private void ribbonOrbOptionButton2_Click_1(object sender, EventArgs e)
        {
            LogFilePathDialog();
        }

        private void ribbonOrbOptionButton1_Click(object sender, EventArgs e)
        {
            LogTypeDialog();
        }

        private void ribbonButtonReload_Click(object sender, EventArgs e)
        {
            ReFreshElements();
        }

        private void ribbonButtonMarkieren_Click(object sender, EventArgs e)
        {
            rtSearch_TextBoxTextChanged(sender, e);

        }

        private void ribbonBtnLoadNew_Click(object sender, EventArgs e)
        {
            ReLoadElements();
        }

        private void ribbonCheckBox1_Click(object sender, EventArgs e)
        {
            
            
        }

        private void ribbonCheckBox1_CheckBoxCheckChanged(object sender, EventArgs e)
        {
            olv.UseAlternatingBackColors = ribbonCheckBox1.Checked;
            olv.Invalidate();
        }

        private void ribbonCheckBox2_CheckBoxCheckChanged(object sender, EventArgs e)
        {
            olv.GridLines = ribbonCheckBox2.Checked;
        }

        private void ribbonCheckBox3_CheckBoxCheckChanged(object sender, EventArgs e)
        {
            olv.FullRowSelect = ribbonCheckBox3.Checked;
        }

        private void kopierenToolStripMenuItem_Click(object sender, EventArgs e)
        {
            this.txtDetails.Copy();
        }

        private void headlineAusfilternToolStripMenuItem_Click(object sender, EventArgs e)
        {
            
            
        }

        private void aToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (olv.SelectedObjects != null)
            {
                foreach (var obj in olv.SelectedObjects)
                {
                    var item = (MSG)obj;
                    if (!(FilterSet.ExcludedHeadLines.Contains(item.Msg1stL)))
                    {
                        this.FilterSet.ExcludedHeadLines.Add(item.Msg1stL);

                        var i = new RibbonButton();
                        i.Text = item.Msg1stL;
                        i.ToolTip = "Klicken um den Filter zu entfernen";
                        i.Click += removeHeadLineFilter;

                        this.ribbonComboHeadFilter.DropDownItems.Add(i);
                    }
                }
               ApplyFilterSet();
            }
        }

        private void quelleAusfilternToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (olv.SelectedObjects != null)
            {
                foreach (var obj in olv.SelectedObjects)
                {
                    var item = (MSG)obj;
                    if (!(FilterSet.ExcludedSourceFiles.Contains(item.Source)))
                    {
                        this.FilterSet.ExcludedSourceFiles.Add(item.Source);

                        var i = new RibbonButton();
                        i.Text = item.Source;
                        i.ToolTip = "Klicken um den Filter zu entfernen";
                        i.Click += removeSourceFilter;

                        this.ribbonComboFilterSource.DropDownItems.Add(i);
                    }
                }
                ApplyFilterSet();
            }
        }

        private void kategorieAusfilternToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (olv.SelectedObjects != null)
            {
                foreach (var obj in olv.SelectedObjects)
                {
                    var item = (MSG)obj;
                    if (!(FilterSet.ExcludedKategories.Contains(item.Kat)))
                    {
                        this.FilterSet.ExcludedKategories.Add(item.Kat);

                        var i = new RibbonButton();
                        i.Text = item.Kat;
                        i.ToolTip = "Klicken um den Filter zu entfernen";
                        i.Click += removeKategorieFilter;

                        this.ribbonComboFilterKateg.DropDownItems.Add(i);
                    }
                }
                ApplyFilterSet();
            }
        }

        private void klasseAusfilternToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (olv.SelectedObjects != null)
            {
                foreach (var obj in olv.SelectedObjects)
                {
                    var item = (MSG)obj;
                    if (!(FilterSet.ExcludedClasses.Contains(item.Klasse)))
                    {
                        this.FilterSet.ExcludedClasses.Add(item.Klasse);

                        var i = new RibbonButton();
                        i.Text = item.Klasse;
                        i.ToolTip = "Klicken um den Filter zu entfernen";
                        i.Click += removeClassFilter;

                        this.ribbonComboFilterKlasse.DropDownItems.Add(i);
                    }
                }
                ApplyFilterSet();
            }
        }
        
        private void olv_KeyUp(object sender, KeyEventArgs e)
        {
            LoadSelectedObjectDetails();
        }

        private void frmMain_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (currentLogSet != null)
                if (MessageBox.Show("Wollen Sie vor dem Beenden speichern?", "Speichern?", MessageBoxButtons.YesNo,MessageBoxIcon.Question) == DialogResult.Yes)
                    save();

            recentOpenedFiles.saveRecentFiles();
        }

        private void RefreshMarkColor()
        {
            this.ribbonButtonMark1.SmallImage = getColoredImage(Color1.R,Color1.G,Color1.B);
            this.ribbonButtonMark2.SmallImage = getColoredImage(Color2.R, Color2.G, Color2.B);
            this.ribbonButtonMark3.SmallImage = getColoredImage(Color3.R, Color3.G, Color3.B);

            colorizeCellCheck();
        }
        private Image getColoredImage(int r, int g, int b)
        {
            var p1 = new Bitmap(16, 16);
            Graphics drawing = Graphics.FromImage(p1);
            var brush = new SolidBrush(Color.FromArgb(r, g, b));

            drawing.FillRectangle(brush, 0, 0, 16, 16);

            drawing.Save();
            drawing.Dispose();

            return p1;
        }

        private void olv_FormatRow(object sender, FormatRowEventArgs e)
        {
            var item = (MSG)e.Model;

            switch (item.MarkerID)
            {
                case 1:
                    e.Item.BackColor = Color1;
                    e.Item.ForeColor = Color.White;
                    break;
                case 2:
                    e.Item.BackColor = Color2;
                    break;
                case 3:
                    e.Item.BackColor = Color3;
                    e.Item.ForeColor = Color.White;
                    break;

            }
        }
        
        private void ribbonButtonMark1_Click(object sender, EventArgs e)
        {
            MarkSelectedColor(1);
        }

        private void ribbonButtonMark2_Click(object sender, EventArgs e)
        {
            MarkSelectedColor(2);
        }

        private void ribbonButtonMark3_Click(object sender, EventArgs e)
        {
            MarkSelectedColor(3);
        }

        private void olv_SelectionChanged(object sender, EventArgs e)
        {
            LoadSelectionMarkerState();
        }

        private void ribbonCheckSelectC1_CheckBoxCheckChanged(object sender, EventArgs e)
        {
            ApplyFilterSet();
        }

        private void ribbonCheckSelectC2_CheckBoxCheckChanged(object sender, EventArgs e)
        {
            ApplyFilterSet();
        }

        private void ribbonCheckSelectC3_CheckBoxCheckChanged(object sender, EventArgs e)
        {
            ApplyFilterSet();
        }

        private void ribbonCheckOnlyToday_CheckBoxCheckChanged(object sender, EventArgs e)
        {
            ApplyFilterSet();
        }

        private void alsTextAusgebenToolStripMenuItem_Click(object sender, EventArgs e)
        {
            ExportSelectedToText();

        }

        private void ColorizeCells(object sender, FormatCellEventArgs e)
        {
            if ((ribbonCheckUseColorCellsKat.Checked) && (e.ColumnIndex == this.olvColumn3.Index))
            {
                MSG item = (MSG)e.Model;
                if (item.Kat.Trim().ToUpper().Equals("ERROR"))
                    e.SubItem.ForeColor = Color.Red;

                if (item.Kat.Trim().ToUpper().Equals("WARN"))
                    e.SubItem.ForeColor = Color.Orange;
            }

            if ((ribbonCheckColorThread.Checked) && (e.ColumnIndex == this.olvColumn4.Index))
            {
                MSG item = (MSG)e.Model;
                Color c;
                if (HashColors.TryGetValue(item.Thread,out c))
                    e.SubItem.ForeColor = c;
            }

            if ((ribbonCheckColorClass.Checked) && (e.ColumnIndex == this.olvColumn5.Index))
            {
                MSG item = (MSG)e.Model;
                Color c;
                if (HashColors.TryGetValue(item.Klasse, out c))
                    e.SubItem.ForeColor = c;
            }
            if ((ribbonCheckColorSource.Checked) && (e.ColumnIndex == this.olvColumn7.Index))
            {
                MSG item = (MSG)e.Model;
                Color c;
                if (HashColors.TryGetValue(item.Source, out c))
                    e.SubItem.ForeColor = c;
            }

        }

        private void CalculateHashColors()
        {
            if (lastLoadedObjects == null)
                return;

            HashColors.Clear();

            List<String> threads = lastLoadedObjects.Select(x => x.Thread).Distinct().ToList();
            List<String> classes = lastLoadedObjects.Select(x => x.Klasse).Distinct().ToList();
            List<String> sources = lastLoadedObjects.Select(x => x.Source).Distinct().ToList();

            var md5 = System.Security.Cryptography.MD5.Create();

            foreach (var item in threads)
            {
                /*var hash = item.GetHashCode().ToString().ToCharArray();
                if (hash.Length < 9)
                    return;
                int r = (hash[0] * 100 + hash[1] * 10 * hash[2]) % 255;
                int g = (hash[3] * 100 + hash[4] * 10 * hash[5]) % 255;
                int b = (hash[6] * 100 + hash[7] * 10 * hash[8]) % 255;
                */
                
                var hash = md5.ComputeHash(Encoding.UTF8.GetBytes(item));
                if (hash.Length < 9)
                    continue;
                int r = (hash[0] * 1) % 255; // (hash[0] * 100 + hash[1] * 10 + hash[2]) % 255;
                int g = (hash[1] * 1) % 255; //(hash[3] * 100 + hash[4] * 10 + hash[5]) % 255;
                int b = (hash[2] * 2) % 255; //(hash[6] * 100 + hash[7] * 10 + hash[8]) % 255;

                int lightness = (int)Math.Sqrt((0.299 * (double)r * (double)r + 0.587 * (double)g * (double)g + 0.114 * (double)b * (double)b));

                if (lightness > 128)
                {
                    r = 255 - r;
                    g = 255 - g;
                    b = 255 - b;
                }

                this.HashColors.Add(item, Color.FromArgb(r, g, b));
            }

            foreach (var item in classes)
            {
                /*var hash = item.GetHashCode().ToString().ToCharArray();
                if (hash.Length < 9)
                    continue;
                int r = (hash[0] * 100 + hash[1] * 10 + hash[2]) % 255;
                int g = (hash[3] * 100 + hash[4] * 10 + hash[5]) % 255;
                int b = (hash[6] * 100 + hash[7] * 10 + hash[8]) % 255;
                */
                var hash = md5.ComputeHash(Encoding.UTF8.GetBytes(item));
                if (hash.Length < 9)
                    continue;
                int r = (hash[0] * 1) % 255; // (hash[0] * 100 + hash[1] * 10 + hash[2]) % 255;
                int g = (hash[1] * 1) % 255; //(hash[3] * 100 + hash[4] * 10 + hash[5]) % 255;
                int b = (hash[2] * 2) % 255; //(hash[6] * 100 + hash[7] * 10 + hash[8]) % 255;

                int lightness = (int)Math.Sqrt((0.299 * (double)r * (double)r + 0.587 * (double)g * (double)g + 0.114 * (double)b * (double)b));

                if (lightness > 128)
                {
                    r = 255 - r;
                    g = 255 - g;
                    b = 255 - b;
                }

                this.HashColors.Add(item, Color.FromArgb(r, g, b));
            }

            foreach (var item in sources)
            {

                //var hash = item.GetHashCode().ToString().ToCharArray();

                var hash = md5.ComputeHash(Encoding.UTF8.GetBytes(item));
                if (hash.Length < 9)
                    continue;
                int r = (hash[0]*1)%255; // (hash[0] * 100 + hash[1] * 10 + hash[2]) % 255;
                int g = (hash[1]*1)%255; //(hash[3] * 100 + hash[4] * 10 + hash[5]) % 255;
                int b = (hash[2]*2)%255; //(hash[6] * 100 + hash[7] * 10 + hash[8]) % 255;

                int lightness = (int)Math.Sqrt((0.299 * (double)r * (double)r + 0.587 * (double)g * (double)g + 0.114 * (double)b * (double)b));

                if (lightness > 128)
                {
                    r = 255 - r;
                    g = 255 - g;
                    b = 255 - b;
                }

                this.HashColors.Add(item, Color.FromArgb(r, g, b));
            }
            
        }

        private void ribbonCheckUseColorCells_CheckBoxCheckChanged(object sender, EventArgs e)
        {
            colorizeCellCheck();

        }

        private void ribbonCheckColorThread_CheckBoxCheckChanged(object sender, EventArgs e)
        {
            colorizeCellCheck();
        }

        private void ribbonCheckColorClass_CheckBoxCheckChanged(object sender, EventArgs e)
        {
            colorizeCellCheck();
        }

        private void ribbonCheckColorSource_CheckBoxCheckChanged(object sender, EventArgs e)
        {
            colorizeCellCheck();
        }

        private void statisticToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (olv.SelectedObjects != null)
                if (olv.SelectedObjects.Count == 1)
                {
                    List<MSG> liste;
                    if ((this.lastFilteredObjects != null) && (this.lastFilteredObjects.Count > 0))
                        liste = lastFilteredObjects;
                    else
                        liste = lastLoadedObjects;

                    String StatisticText="";
                    var current = (MSG)olv.SelectedObject;

                    List<MSG> allElem = liste.Where(x => x.Msg1stL == current.Msg1stL).ToList();
                    var anz = allElem.Count();


                    StatisticText += "Auswertung" + Environment.NewLine;
                    StatisticText += String.Format("Fehler vom {0} um {1}", current.TS.ToShortDateString(), current.TS.ToShortTimeString()) + Environment.NewLine;
                    StatisticText += "Headline: " + current.Msg1stL + Environment.NewLine + Environment.NewLine;
                    StatisticText += String.Format("Die Meldung ist insgesamt {0} mal aufgetreten",anz) + Environment.NewLine+Environment.NewLine;

                    StatisticText += String.Format("Montags:\t{0}",allElem.Where(x=>x.TS.DayOfWeek==DayOfWeek.Monday).Count()) + Environment.NewLine;
                    StatisticText += String.Format("Dienstags:\t{0}", allElem.Where(x => x.TS.DayOfWeek == DayOfWeek.Tuesday).Count()) + Environment.NewLine;
                    StatisticText += String.Format("Mittwochs:\t{0}", allElem.Where(x => x.TS.DayOfWeek == DayOfWeek.Wednesday).Count()) + Environment.NewLine;
                    StatisticText += String.Format("Donnerstags:\t{0}", allElem.Where(x => x.TS.DayOfWeek == DayOfWeek.Thursday).Count()) + Environment.NewLine;
                    StatisticText += String.Format("Freitags:\t{0}", allElem.Where(x => x.TS.DayOfWeek == DayOfWeek.Friday).Count()) + Environment.NewLine;
                    StatisticText += String.Format("Samstags:\t{0}", allElem.Where(x => x.TS.DayOfWeek == DayOfWeek.Saturday).Count()) + Environment.NewLine;
                    StatisticText += String.Format("Sonntags:\t{0}", allElem.Where(x => x.TS.DayOfWeek == DayOfWeek.Sunday).Count()) + Environment.NewLine;


                    var f = new frmText();
                    f.Text = String.Format("LOG-Text {0:yyyy-MM-dd HH:mm:ss,fff}", current.TS);

                    f.setText(StatisticText);
                    f.Show();

                    

                }
        }

        private void olv_KeyPress(object sender, KeyPressEventArgs e)
        {
            if (e.KeyChar == '1')
            {
                ribbonButtonMark1_Click(sender, e);
                e.Handled = true;
            }

            if (e.KeyChar == '2')
            {
                ribbonButtonMark2_Click(sender, e);
                e.Handled = true;
            }

            if (e.KeyChar == '3')
            {
                ribbonButtonMark3_Click(sender, e);
                e.Handled = true;
            }
            

        }

        private void ribbonButtonExportTXT_Click(object sender, EventArgs e)
        {
            ExportSelectedToText();
        }

        private void ribbonButtonExportCSV_Click(object sender, EventArgs e)
        {
            ExportSelectedToCSV();
        }

        private void ribbonOrbOptionOpts_Click(object sender, EventArgs e)
        {
            if (currentLogSet != null)
            {
                var f = new frmOpt();
                f.SetOptions(currentLogSet.CheckFileMaxAge, currentLogSet.FileMaxAge, currentLogSet.SameTimeValue);
                if (f.ShowDialog()==DialogResult.OK)
                {
                    currentLogSet.CheckFileMaxAge = f.GetParseOnlyLatest();
                    currentLogSet.FileMaxAge = f.GetDaysLatest();
                    currentLogSet.SameTimeValue = f.GetSameTimeValue();
                }
            }
        }

        private void ribbonOrbOptionButton3_Click(object sender, EventArgs e)
        {
            var f = new frmInfo(VERSION);
            f.Show();
        }

        private void ribbonButtonReloadFilter_Click(object sender, EventArgs e)
        {
            ReLoadElements(false);
        }

        private void ribbonOrbMenuUpdate_Click(object sender, EventArgs e)
        {
            CheckUpdates();
        }

        private void ribbonOrbMenuNew_Click(object sender, EventArgs e)
        {
            NewFileSet();
        }
    }
}
